<# :
@echo off
REM ----------------------------------------------------------------------------
REM METALZINHO.COM
REM HYBRID SCRIPT: This section runs in Batch to launch PowerShell safely.
REM We use this method to bypass the character encoding issues (Â, Ã, etc.)
REM that cause standard .bat files to fail in folders like "LÂMINA".
REM ----------------------------------------------------------------------------
setlocal
cd /d "%~dp0"
powershell -NoLogo -NoProfile -ExecutionPolicy Bypass -Command "Invoke-Command -ScriptBlock ([ScriptBlock]::Create((Get-Content -Raw '%~f0')))"
pause
exit /b
#>

# ==============================================================================
# POWERSHELL LOGIC STARTS HERE (This handles the heavy lifting)
# ==============================================================================

$scriptPath = Get-Location
Write-Host "==============================================================" -ForegroundColor Cyan
Write-Host " INICIANDO PROCESSAMENTO (Calculo de Parede)" -ForegroundColor Cyan
Write-Host " Raiz: $scriptPath" -ForegroundColor Gray
Write-Host "=============================================================="

# 1. Find all 'Results.csv' files recursively
# ------------------------------------------------------------------------------
$foundFiles = Get-ChildItem -Path $scriptPath -Filter "Results.csv" -Recurse -ErrorAction SilentlyContinue

if ($foundFiles.Count -eq 0) {
    Write-Host " [ERRO] Nenhum arquivo 'Results.csv' encontrado." -ForegroundColor Red
    exit
}

Write-Host " -> Encontrados $($foundFiles.Count) arquivos para processar." -ForegroundColor White

foreach ($file in $foundFiles) {
    $dir = $file.DirectoryName
    $outputFile = Join-Path -Path $dir -ChildPath "medidas parede2.csv"
    
    Write-Host ""
    $relativePath = $dir.Substring($scriptPath.Path.Length)
    Write-Host " Pasta: ..$relativePath" -ForegroundColor Yellow

    # FAILSAFE: Don't overwrite existing files
    if (Test-Path $outputFile) {
        Write-Host "    [PULADO] 'medidas parede2.csv' ja existe." -ForegroundColor DarkGray
        continue
    }

    try {
        # 2. Read CSV
        # --------------------------------------------------------------------------
        $csv = Import-Csv $file.FullName
        
        if ($csv.Count -eq 0) {
            Write-Host "    [FALHA] Arquivo vazio." -ForegroundColor Red
            continue
        }

        # 3. Detect Data Column
        # --------------------------------------------------------------------------
        $cols = $csv[0].PSObject.Properties.Name
        $idx = $cols[0] 
        
        # Find first numerical column
        $measure = $cols | Where-Object {$_ -ne $idx} | Where-Object { ($csv[0].$_ -as [double]) -ne $null } | Select-Object -First 1

        if (-not $measure) {
            Write-Host "    [FALHA] Coluna de medidas nao identificada." -ForegroundColor Red
            continue
        }

        # 4. Transform & Calculate: (Diametro - Lumen) / 2
        # --------------------------------------------------------------------------
        $out = @()
        
        # Use InvariantCulture to ensure '.' is treated as decimal separator during math
        $culture = [System.Globalization.CultureInfo]::InvariantCulture

        for ($i = 0; $i -lt $csv.Count; $i += 2) {
            if ($i + 1 -lt $csv.Count) {
                # Get raw strings
                $rawDiametro = $csv[$i].$measure
                $rawLumen    = $csv[$i+1].$measure
                
                # Convert to numbers for math (Robust parsing)
                try {
                    $valDiametro = [double]::Parse($rawDiametro, $culture)
                    $valLumen    = [double]::Parse($rawLumen, $culture)
                } catch {
                    $valDiametro = 0
                    $valLumen = 0
                }

                # PERFORM CALCULATION: Parede = (Diametro - Lumen) / 2
                $valParede = ($valDiametro - $valLumen) / 2
                
                # Create object
                $obj = [PSCustomObject]@{
                    diametro = $valDiametro
                    lumen    = $valLumen
                    parede   = $valParede
                }
                $out += $obj
            }
        }

        # 5. Export (Replace dots with commas for ALL columns)
        # --------------------------------------------------------------------------
        $out | ForEach-Object {
            # Force conversion to string and replace dot with comma
            $_.diametro = "$($_.diametro)".Replace('.', ',')
            $_.lumen    = "$($_.lumen)".Replace('.', ',')
            $_.parede   = "$($_.parede)".Replace('.', ',')
            $_
        } | Export-Csv -Path $outputFile -NoTypeInformation -Delimiter ";" -Encoding UTF8

        Write-Host "    [SUCESSO] Gerado com $( $out.Count ) linhas. (Calculo efetuado)" -ForegroundColor Green

    } catch {
        Write-Host "    [ERRO CRITICO] $($_.Exception.Message)" -ForegroundColor Red
    }
}

Write-Host "`n==============================================================" -ForegroundColor Cyan
Write-Host " CONCLUIDO." -ForegroundColor Cyan